##############################################################################
# OpenECHO
#
# Credit card transactions via SSL to
# Electronic Clearing House (ECHO) Systems
#
# Refer to ECHO's documentation for more info
# https://wwws.echo-inc.com
#
# AUTHOR
# Michael Lehmkuhl <michael@electricpulp.com>
#
# SPECIAL THANKS
# Jim Darden <support@openecho.com>
# Dan Browning <db@kavod.com>
#
# VERSION HISTORY
# + v1.2 08/17/2002 Corrected problem with certain string comparisons.
# + v1.3 08/23/2002 Converted Interchange GlobalSub to Vend::Payment module.
#
# Copyright (C) 2002 Electric Pulp. <info@electricpulp.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this program; if not, write to the Free
# Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA  02111-1307  USA.
#
################################################################################

package OpenECHO;

BEGIN {

	my $selected;
	eval {
		require Net::SSLeay;
		import Net::SSLeay qw(get_https make_form make_headers);
		$selected = "Net::SSLeay";
	};

	$OpenECHO::Have_Net_SSLeay = 1 unless $@;

	unless ($OpenECHO::Have_Net_SSLeay) {

		eval {
			require LWP::UserAgent;
			require HTTP::Request::Common;
			require Crypt::SSLeay;
			import HTTP::Request::Common qw(GET);
			$selected = "LWP and Crypt::SSLeay";
		};

		$OpenECHO::Have_LWP = 1 unless $@;

	}

	unless ($OpenECHO::Have_Net_SSLeay or $OpenECHO::Have_LWP) {
		die "OpenECHO requires Net::SSLeay, or LWP and Crypt::SSLeay";
	}

	use URI::Escape;

}

sub new {
	my $class = shift;
	my $this = {};
	bless $this, $class;
	$this->_initialize();
	return $this;
}

sub Submit {
	my $this = shift;

	if ($this->{'EchoServer'}) {
		$URL = $this->{'EchoServer'};
	}
	else {
		$URL = "https://wwws.echo-inc.com/scripts/INR200.EXE";
	}

	$this->{'EchoResponse'} = "";

	$data = $this->getURLData();

	if ($OpenECHO::Have_Net_SSLeay) {
		require Net::SSLeay;
		import Net::SSLeay qw(get_https make_form make_headers);

		my ($server, $port, $script) = $URL =~ m"^https://([^/]+)(:[0-9]+)?(/.*?)$";
		($port) = $port =~ s":"";
		$port = '443' unless $port;

		$script .= '?'.$data if $data;

		my ($page, $response, %reply_headers) = get_https($server, $port, $script);
		$this->{'EchoResponse'} = $page;
	}
	elsif ($OpenECHO::Have_LWP) {
		# NOTE: This method is less tested and has been reported by some to be somewhat buggy.
		#       Net::SSLeay is preferred.

		require LWP::UserAgent;
		import LWP::UserAgent;

		my $ua = LWP::UserAgent->new;
		my $req = HTTP::Request->new(GET => $URL);
		my $res = $ua->request($req);
		if ($res->is_success) {
			$this->{'EchoResponse'} = $res->as_string;
		}
		else {
			die "Failed: ", $res->status_line, "\n";
		}
	}


	$this->{'echotype1'} = $this->GetEchoReturn($this->{'EchoResponse'}, 1);
	$this->{'echotype2'} = $this->GetEchoReturn($this->{'EchoResponse'}, 2);
	$this->{'echotype3'} = $this->GetEchoReturn($this->{'EchoResponse'}, 3);
	$this->{'openecho'} = $this->GetEchoReturn($this->{'EchoResponse'}, 'OPEN');

	# Get all the metadata.
	$this->{'auth_code'} = $this->GetEchoProp($this->{'echotype3'}, "auth_code");
	$this->{'order_number'} = $this->GetEchoProp($this->{'echotype3'}, "order_number");
	$this->{'reference'} = $this->GetEchoProp($this->{'echotype3'}, "echo_reference");
	$this->{'status'} = $this->GetEchoProp($this->{'echotype3'}, "status");
	$this->{'avs_result'} = $this->GetEchoProp($this->{'echotype3'}, "avs_result");
	$this->{'security_result'} = $this->GetEchoProp($this->{'echotype3'}, "security_result");
	$this->{'mac'} = $this->GetEchoProp($this->{'echotype3'}, "mac");
	$this->{'cs_factors'} = $this->GetEchoProp($this->{'echotype3'}, "cs_factors");
	$this->{'cs_flag'} = $this->GetEchoProp($this->{'echotype3'}, "cs_flag");
	$this->{'cs_host_score'} = $this->GetEchoProp($this->{'echotype3'}, "cs_host_score");
	$this->{'cs_reference_number'} = $this->GetEchoProp($this->{'echotype3'}, "cs_reference_number");
	$this->{'cs_response'} = $this->GetEchoProp($this->{'echotype3'}, "cs_response");
	$this->{'cs_score'} = $this->GetEchoProp($this->{'echotype3'}, "cs_score");
	$this->{'cs_status'} = $this->GetEchoProp($this->{'echotype3'}, "cs_status");


	if ($this->{'transaction_type'} eq "AD") {
		if ($this->{'avs_result'} eq "X" or $this->{'avs_result'} eq "Y") {
			$this->{'EchoSuccess'} = 1;
		} else {
			$this->{'EchoSuccess'} = 0;
		}
	} else {
		$this->{'EchoSuccess'} = !($this->{'status'} eq "D");
	}


	if ($this->{'EchoResponse'} eq "") {
		$this->{'EchoSuccess'} = 0;
	}

	# make sure we assign an integer to EchoSuccess
	($this->{'EchoSuccess'}) ? ($this->{'EchoSuccess'} = 1) : ($this->{'EchoSuccess'} = 0);

	return $this->{'EchoSuccess'};

} # sub Submit


sub getURLData {
	my $this = shift;

	my $s =
	"order_type=" 				. $this->{'order_type'} .
	"&transaction_type=" 		. $this->{'transaction_type'} .
	"&merchant_echo_id=" 		. $this->{'merchant_echo_id'} .
	"&merchant_pin=" 			. $this->{'merchant_pin'} .
	"&isp_echo_id=" 			. $this->{'isp_echo_id'} .
	"&isp_pin=" 				. $this->{'isp_pin'} .
	"&authorization=" 			. $this->{'authorization'} .
	"&billing_ip_address=" 		. $this->{'billing_ip_address'} .
	"&billing_prefix="			. $this->{'billing_prefix'} .
	"&billing_name="			. $this->{'billing_name'} .
	"&billing_first_name="		. $this->{'billing_first_name'} .
	"&billing_last_name="		. $this->{'billing_last_name'} .
	"&billing_company_name="	. $this->{'billing_company_name'} .
	"&billing_address1="		. $this->{'billing_address1'} .
	"&billing_address2="		. $this->{'billing_address2'} .
	"&billing_city="			. $this->{'billing_city'} .
	"&billing_state="			. $this->{'billing_state'} .
	"&billing_zip="				. $this->{'billing_zip'} .
	"&billing_country="			. $this->{'billing_country'} .
	"&billing_phone="			. $this->{'billing_phone'} .
	"&billing_fax="				. $this->{'billing_fax'} .
	"&billing_email="			. $this->{'billing_email'} .
	"&cc_number="				. $this->{'cc_number'} .
	"&ccexp_month="				. $this->{'ccexp_month'} .
	"&ccexp_year="				. $this->{'ccexp_year'} .
	"&counter="					. $this->{'counter'} .
	"&debug="					. $this->{'debug'};

	if (($this->{'transaction_type'} eq "DD") ||
		($this->{'transaction_type'} eq "DC") ||
		($this->{'transaction_type'} eq "DV")) {
		$s .=
		"&ec_account="				. $this->{'ec_account'} .
		"&ec_account_type="			. $this->{'ec_account_type'} .
		"&ec_payment_type="			. $this->{'ec_payment_type'} .
		"&ec_address1="				. $this->{'ec_address1'} .
		"&ec_address2="				. $this->{'ec_address2'} .
		"&ec_bank_name="			. $this->{'ec_bank_name'} .
		"&ec_business_acct="		. $this->{'ec_business_acct'} .
		"&ec_city="					. $this->{'ec_city'} .
		"&ec_email="				. $this->{'ec_email'} .
		"&ec_first_name="			. $this->{'ec_first_name'} .
		"&ec_id_country="			. $this->{'ec_id_country'} .
		"&ec_id_exp_mm="			. $this->{'ec_id_exp_mm'} .
		"&ec_id_exp_dd="			. $this->{'ec_id_exp_dd'} .
		"&ec_id_exp_yy="			. $this->{'ec_id_exp_yy'} .
		"&ec_id_number="			. $this->{'ec_id_number'} .
		"&ec_id_state="				. $this->{'ec_id_state'} .
		"&ec_id_type="				. $this->{'ec_id_type'} .
		"&ec_last_name="			. $this->{'ec_last_name'} .
		"&ec_merchant_ref="			. $this->{'ec_merchant_ref'} .
		"&ec_nbds_code="			. $this->{'ec_nbds_code'} .
		"&ec_other_name="			. $this->{'ec_other_name'} .
		"&ec_payee="				. $this->{'ec_payee'} .
		"&ec_rt="					. $this->{'ec_rt'} .
		"&ec_serial_number="		. $this->{'ec_serial_number'} .
		"&ec_state="				. $this->{'ec_state'} .
		"&ec_transaction_dt="		. $this->{'ec_transaction_dt'} .
		"&ec_zip="					. $this->{'ec_zip'};
	}

	$s .=
	"&grand_total="				. $this->{'grand_total'} .
	"&merchant_email="			. $this->{'merchant_email'} .
	"&merchant_trace_nbr="		. $this->{'merchant_trace_nbr'} .
	"&original_amount="			. $this->{'original_amount'} .
	"&original_trandate_mm="	. $this->{'original_trandate_mm'} .
	"&original_trandate_dd="	. $this->{'original_trandate_dd'} .
	"&original_trandate_yyyy="	. $this->{'original_trandate_yyyy'} .
	"&original_reference="		. $this->{'original_reference'} .
	"&order_number="			. $this->{'order_number'} .
	"&shipping_flag="			. $this->{'shipping_flag'} .
	"&shipping_prefix="			. $this->{'shipping_prefix'} .
	"&shipping_name="			. $this->{'shipping_name'} .
	"&shipping_address1="		. $this->{'shipping_address1'} .
	"&shipping_address2="		. $this->{'shipping_address2'} .
	"&shipping_city="			. $this->{'shipping_city'} .
	"&shipping_state="			. $this->{'shipping_state'} .
	"&shipping_zip="			. $this->{'shipping_zip'} .
	"&shipping_comments="		. $this->{'shipping_comments'} .
	"&shipping_country="		. $this->{'shipping_country'} .
	"&shipping_phone="			. $this->{'shipping_phone'} .
	"&shipping_fax="			. $this->{'shipping_fax'} .
	"&shipper="					. $this->{'shipper'} .
	"&shipper_tracking_nbr="	. $this->{'shipper_tracking_nbr'} .
	"&track1="					. $this->{'track1'} .
	"&cnp_security="			. $this->{'cnp_security'} .
	"&cnp_recurring="			. $this->{'cnp_recurring'};

	return $s;

} # end getURLData



#################################################
# All the get/set methods for the echo properties
#################################################
sub set_order_type {
	my $this = shift;
	my $value = shift;
	$this->{'order_type'} = $value;
}

sub get_order_type {
	my $this = shift;
	return $this->{'order_type'};
}

sub set_transaction_type {
	my $this = shift;
	my $value = shift;
	$this->{'transaction_type'} = $value;
}

sub get_transaction_type {
	my $this = shift;
	return $this->{'transaction_type'};
}

sub set_merchant_echo_id {
	my $this = shift;
	my $value = shift;
	$this->{'merchant_echo_id'} = uri_escape($value);
}

sub get_merchant_echo_id {
	my $this = shift;
	return $this->{'merchant_echo_id'};
}

sub set_merchant_pin {
	my $this = shift;
	my $value = shift;
	$this->{'merchant_pin'} = uri_escape($value);
}

sub get_merchant_pin {
	my $this = shift;
	return $this->{'merchant_pin'};
}

sub set_isp_echo_id {
	my $this = shift;
	my $value = shift;
	$this->{'isp_echo_id'} = uri_escape($value);
}

sub get_isp_echo_id {
	my $this = shift;
	return $this->{'isp_echo_id'};
}

sub set_isp_pin {
	my $this = shift;
	my $value = shift;
	$this->{'isp_pin'} = uri_escape($value);
}

sub get_isp_pin {
	my $this = shift;
	return $this->{'isp_pin'};
}

sub set_authorization {
	my $this = shift;
	my $value = shift;
	$this->{'authorization'} = $value;
}

sub get_authorization {
	my $this = shift;
	return $this->{'authorization'};
}

sub set_billing_ip_address {
	my $this = shift;
	my $value = shift;
	$this->{'billing_ip_address'} = $value;
}

sub get_billing_ip_address {
	my $this = shift;
	return $this->{'billing_ip_address'};
}

sub set_billing_prefix {
	my $this = shift;
	my $value = shift;
	$this->{'billing_prefix'} = uri_escape($value);
}

sub get_billing_prefix {
	my $this = shift;
	return $this->{'billing_prefix'};
}

sub set_billing_name {
	my $this = shift;
	my $value = shift;
	$this->{'billing_name'} = uri_escape($value);
}

sub get_billing_name {
	my $this = shift;
	return $this->{'billing_name'};
}

sub set_billing_first_name {
	my $this = shift;
	my $value = shift;
	$this->{'billing_first_name'} = uri_escape($value);
}

sub get_billing_first_name {
	my $this = shift;
	return $this->{'billing_first_name'};
}
sub set_billing_last_name {
	my $this = shift;
	my $value = shift;
	$this->{'billing_last_name'} = uri_escape($value);
}

sub get_billing_last_name {
	my $this = shift;
	return $this->{'billing_last_name'};
}
sub set_billing_company_name {
	my $this = shift;
	my $value = shift;
	$this->{'billing_company_name'} = uri_escape($value);
}

sub get_billing_company_name {
	my $this = shift;
	return $this->{'billing_company_name'};
}

sub set_billing_address1 {
	my $this = shift;
	my $value = shift;
	$this->{'billing_address1'} = uri_escape($value);
}

sub get_billing_address1 {
	my $this = shift;
	return $this->{'billing_address1'};
}

sub set_billing_address2 {
	my $this = shift;
	my $value = shift;
	$this->{'billing_address2'} = uri_escape($value);
}

sub get_billing_address2 {
	my $this = shift;
	return $this->{'billing_address2'};
}

sub set_billing_city {
	my $this = shift;
	my $value = shift;
	$this->{'billing_city'} = uri_escape($value);
}

sub get_billing_city {
	my $this = shift;
	return $this->{'billing_city'};
}

sub set_billing_state {
	my $this = shift;
	my $value = shift;
	$this->{'billing_state'} = uri_escape($value);
}

sub get_billing_state {
	my $this = shift;
	return $this->{'billing_state'};
}

sub set_billing_zip {
	my $this = shift;
	my $value = shift;
	$this->{'billing_zip'} = uri_escape($value);
}

sub get_billing_zip {
	my $this = shift;
	return $this->{'billing_zip'};
}

sub set_billing_country {
	my $this = shift;
	my $value = shift;
	$this->{'billing_country'} = uri_escape($value);
}

sub get_billing_country {
	my $this = shift;
	return $this->{'billing_country'};
}

sub set_billing_phone {
	my $this = shift;
	my $value = shift;
	$this->{'billing_phone'} = uri_escape($value);
}

sub get_billing_phone {
	my $this = shift;
	return $this->{'billing_phone'};
}

sub set_billing_fax {
	my $this = shift;
	my $value = shift;
	$this->{'billing_fax'} = uri_escape($value);
}

sub get_billing_fax {
	my $this = shift;
	return $this->{'billing_fax'};
}

sub set_billing_email {
	my $this = shift;
	my $value = shift;
	$this->{'billing_email'} = uri_escape($value);
}

sub get_billing_email {
	my $this = shift;
	return $this->{'billing_email'};
}

sub set_cc_number {
	my $this = shift;
	my $value = shift;
	$this->{'cc_number'} = uri_escape($value);
}

sub get_cc_number {
	my $this = shift;
	return $this->{'cc_number'};
}

sub set_ccexp_month {
	my $this = shift;
	my $value = shift;
	$this->{'ccexp_month'} = $value;
}

sub get_ccexp_month {
	my $this = shift;
	return $this->{'ccexp_month'};
}

sub set_ccexp_year {
	my $this = shift;
	my $value = shift;
	$this->{'ccexp_year'} = $value;
}

sub get_ccexp_year {
	my $this = shift;
	return $this->{'ccexp_year'};
}

sub set_counter {
	my $this = shift;
	my $value = shift;
	$this->{'counter'} = $value;
}

sub get_counter {
	my $this = shift;
	return $this->{'counter'};
}

sub set_debug {
	my $this = shift;
	my $value = shift;
	$this->{'debug'} = $value;
}

sub get_debug {
	my $this = shift;
	return $this->{'debug'};
}

sub set_ec_account {
	my $this = shift;
	my $value = shift;
	$this->{'ec_account'} = uri_escape($value);
}

sub get_ec_account {
	my $this = shift;
	return $this->{'ec_account'};
}

sub set_ec_account_type {
	my $this = shift;
	my $value = shift;
	$this->{'ec_account_type'} = uri_escape($value);
}

sub get_ec_account_type {
	my $this = shift;
	return $this->{'ec_account_type'};
}

sub set_ec_payment_type {
	my $this = shift;
	my $value = shift;
	$this->{'ec_payment_type'} = uri_escape($value);
}

sub get_ec_payment_type {
	my $this = shift;
	return $this->{'ec_payment_type'};
}

sub set_ec_address1 {
	my $this = shift;
	my $value = shift;
	$this->{'ec_address1'} = uri_escape($value);
}

sub set_ec_address1 {
	my $this = shift;
	my $value = shift;
	$this->{'ec_address1'} = uri_escape($value);
}

sub set_ec_address1 {
	my $this = shift;
	my $value = shift;
	$this->{'ec_address1'} = uri_escape($value);
}

sub get_ec_address1 {
	my $this = shift;
	return $this->{'ec_address1'};
}

sub set_ec_address2 {
	my $this = shift;
	my $value = shift;
	$this->{'ec_address2'} = uri_escape($value);
}

sub get_ec_address2 {
	my $this = shift;
	return $this->{'ec_address2'};
}

sub set_ec_bank_name {
	my $this = shift;
	my $value = shift;
	$this->{'ec_bank_name'} = uri_escape($value);
}

sub get_ec_bank_name {
	my $this = shift;
	return $this->{'ec_bank_name'};
}

sub set_ec_business_acct {
	my $this = shift;
	my $value = shift;
	$this->{'ec_business_acct'} = uri_escape($value);
}

sub get_ec_business_acct {
	my $this = shift;
	return $this->{'ec_business_acct'};
}

sub set_ec_city {
	my $this = shift;
	my $value = shift;
	$this->{'ec_city'} = $value;
}

sub get_ec_city {
	my $this = shift;
	return $this->{'ec_city'};
}

sub set_ec_email {
	my $this = shift;
	my $value = shift;
	$this->{'ec_email'} = uri_escape($value);
}

sub get_ec_email {
	my $this = shift;
	return $this->{'ec_email'};
}

sub set_ec_first_name {
	my $this = shift;
	my $value = shift;
	$this->{'ec_first_name'} = uri_escape($value);
}

sub get_ec_first_name {
	my $this = shift;
	return $this->{'ec_first_name'};
}

sub set_ec_id_country {
	my $this = shift;
	my $value = shift;
	$this->{'ec_id_country'} = uri_escape($value);
}

sub get_ec_id_country {
	my $this = shift;
	return $this->{'ec_id_country'};
}

sub set_ec_id_exp_mm {
	my $this = shift;
	my $value = shift;
	$this->{'ec_id_exp_mm'} = $value;
}

sub get_ec_id_exp_mm {
	my $this = shift;
	return $this->{'ec_id_exp_mm'};
}

sub set_ec_id_exp_dd {
	my $this = shift;
	my $value = shift;
	$this->{'ec_id_exp_dd'} = $value;
}

sub get_ec_id_exp_dd {
	my $this = shift;
	return $this->{'ec_id_exp_dd'};
}

sub set_ec_id_exp_yy {
	my $this = shift;
	my $value = shift;
	$this->{'ec_id_exp_yy'} = $value;
}

sub get_ec_id_exp_yy {
	my $this = shift;
	return $this->{'ec_id_exp_yy'};
}

sub set_ec_id_number {
	my $this = shift;
	my $value = shift;
	$this->{'ec_id_number'} = uri_escape($value);
}

sub get_ec_id_number {
	my $this = shift;
	return $this->{'ec_id_number'};
}

sub set_ec_id_state {
	my $this = shift;
	my $value = shift;
	$this->{'ec_id_state'} = uri_escape($value);
}

sub get_ec_id_state {
	my $this = shift;
	return $this->{'ec_id_state'};
}

sub set_ec_id_type {
	my $this = shift;
	my $value = shift;
	$this->{'ec_id_type'} = $value;
}

sub get_ec_id_type {
	my $this = shift;
	return $this->{'ec_id_type'};
}

sub set_ec_last_name {
	my $this = shift;
	my $value = shift;
	$this->{'ec_last_name'} = uri_escape($value);
}

sub get_ec_last_name {
	my $this = shift;
	return $this->{'ec_last_name'};
}

sub set_ec_merchant_ref {
	my $this = shift;
	my $value = shift;
	$this->{'ec_merchant_ref'} = $value;
}

sub get_ec_merchant_ref {
	my $this = shift;
	return $this->{'ec_merchant_ref'};
}

sub set_ec_nbds_code {
	my $this = shift;
	my $value = shift;
	$this->{'ec_nbds_code'} = $value;
}

sub get_ec_nbds_code {
	my $this = shift;
	return $this->{'ec_nbds_code'};
}

sub set_ec_other_name {
	my $this = shift;
	my $value = shift;
	$this->{'ec_other_name'} = uri_escape($value);
}

sub get_ec_other_name {
	my $this = shift;
	return $this->{'ec_other_name'};
}

sub set_ec_payee {
	my $this = shift;
	my $value = shift;
	$this->{'ec_payee'} = uri_escape($value);
}

sub get_ec_payee {
	my $this = shift;
	return $this->{'ec_payee'};
}

sub set_ec_rt {
	my $this = shift;
	my $value = shift;
	$this->{'ec_rt'} = uri_escape($value);
}

sub get_ec_rt {
	my $this = shift;
	return $this->{'ec_rt'};
}

sub set_ec_serial_number {
	my $this = shift;
	my $value = shift;
	$this->{'ec_serial_number'} = uri_escape($value);
}

sub get_ec_serial_number {
	my $this = shift;
	return $this->{'ec_serial_number'};
}

sub set_ec_state {
	my $this = shift;
	my $value = shift;
	$this->{'ec_state'} = uri_escape($value);
}

sub get_ec_state {
	my $this = shift;
	return $this->{'ec_state'};
}

sub set_ec_transaction_dt {
	my $this = shift;
	my $value = shift;
	$this->{'ec_transaction_dt'} = uri_escape($value);
}

sub get_ec_transaction_dt {
	my $this = shift;
	return $this->{'ec_transaction_dt'};
}

sub set_ec_zip {
	my $this = shift;
	my $value = shift;
	$this->{'ec_zip'} = uri_escape($value);
}

sub get_ec_zip {
	my $this = shift;
	return $this->{'ec_zip'};
}

sub set_grand_total {
	my $this = shift;
	my $value = shift;
	$this->{'grand_total'} = sprintf("%01.2f", $value);
}

sub get_grand_total {
	my $this = shift;
	return $this->{'grand_total'};
}

sub set_merchant_email {
	my $this = shift;
	my $value = shift;
	$this->{'merchant_email'} = uri_escape($value);
}

sub get_merchant_email {
	my $this = shift;
	return $this->{'merchant_email'};
}

sub set_merchant_trace_nbr {
	my $this = shift;
	my $value = shift;
	$this->{'merchant_trace_nbr'} = $value;
}

sub get_merchant_trace_nbr {
	my $this = shift;
	return $this->{'merchant_trace_nbr'};
}

sub set_original_amount {
	my $this = shift;
	my $value = shift;
	$this->{'original_amount'} = sprintf("%01.2f", $value);
}

sub get_original_amount {
	my $this = shift;
	return $this->{'original_amount'};
}

sub set_original_trandate_mm {
	my $this = shift;
	my $value = shift;
	$this->{'original_trandate_mm'} = $value;
}

sub get_original_trandate_mm {
	my $this = shift;
	return $this->{'original_trandate_mm'};
}

sub set_original_trandate_dd {
	my $this = shift;
	my $value = shift;
	$this->{'original_trandate_dd'} = $value;
}

sub get_original_trandate_dd {
	my $this = shift;
	return $this->{'original_trandate_dd'};
}

sub set_original_trandate_yyyy {
	my $this = shift;
	my $value = shift;
	$this->{'original_trandate_yyyy'} = $value;
}

sub get_original_trandate_yyyy {
	my $this = shift;
	return $this->{'original_trandate_yyyy'};
}

sub set_original_reference {
	my $this = shift;
	my $value = shift;
	$this->{'original_reference'} = $value;
}

sub get_original_reference {
	my $this = shift;
	return $this->{'original_reference'};
}

sub set_order_number {
	my $this = shift;
	my $value = shift;
	$this->{'order_number'} = $value;
}

sub get_order_number {
	my $this = shift;
	return $this->{'order_number'};
}

sub set_shipping_flag {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_flag'} = $value;
}

sub get_shipping_flag {
	my $this = shift;
	return $this->{'shipping_flag'};
}

sub set_shipping_prefix {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_prefix'} = uri_escape($value);
}

sub get_shipping_prefix {
	my $this = shift;
	return $this->{'shipping_prefix'};
}

sub set_shipping_name {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_name'} = uri_escape($value);
}

sub get_shipping_name {
	my $this = shift;
	return $this->{'shipping_name'};
}

sub set_shipping_address1 {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_address1'} = uri_escape($value);
}

sub get_shipping_address1 {
	my $this = shift;
	return $this->{'shipping_address1'};
}

sub set_shipping_address2 {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_address2'} = uri_escape($value);
}

sub get_shipping_address2 {
	my $this = shift;
	return $this->{'shipping_address2'};
}

sub set_shipping_city {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_city'} = uri_escape($value);
}

sub get_shipping_city {
	my $this = shift;
	return $this->{'shipping_city'};
}

sub set_shipping_state {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_state'} = uri_escape($value);
}

sub get_shipping_state {
	my $this = shift;
	return $this->{'shipping_state'};
}

sub set_shipping_zip {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_zip'} = uri_escape($value);
}

sub get_shipping_zip {
	my $this = shift;
	return $this->{'shipping_zip'};
}

sub set_shipping_comments {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_comments'} = uri_escape($value);
}

sub get_shipping_comments {
	my $this = shift;
	return $this->{'shipping_comments'};
}

sub set_shipping_country {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_country'} = uri_escape($value);
}

sub get_shipping_country {
	my $this = shift;
	return $this->{'shipping_country'};
}

sub set_shipping_phone {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_phone'} = uri_escape($value);
}

sub get_shipping_phone {
	my $this = shift;
	return $this->{'shipping_phone'};
}

sub set_shipping_fax {
	my $this = shift;
	my $value = shift;
	$this->{'shipping_fax'} = uri_escape($value);
}

sub get_shipping_fax {
	my $this = shift;
	return $this->{'shipping_fax'};
}

sub set_shipper {
	my $this = shift;
	my $value = shift;
	$this->{'shipper'} = uri_escape($value);
}

sub get_shipper {
	my $this = shift;
	return $this->{'shipper'};
}

sub set_shipper_tracking_nbr {
	my $this = shift;
	my $value = shift;
	$this->{'shipper_tracking_nbr'} = $value;
}

sub get_shipper_tracking_nbr {
	my $this = shift;
	return $this->{'shipper_tracking_nbr'};
}

sub set_track1 {
	my $this = shift;
	my $value = shift;
	$this->{'track1'} = uri_escape($value);
}

sub get_track1 {
	my $this = shift;
	return $this->{'track1'};
}

sub set_track2 {
	my $this = shift;
	my $value = shift;
	$this->{'track2'} = uri_escape($value);
}

sub get_track2 {
	my $this = shift;
	return $this->{'track2'};
}

sub set_cnp_recurring {
	my $this = shift;
	my $value = shift;
	$this->{'cnp_recurring'} = $value;
}

sub set_cnp_security {
	my $this = shift;
	my $value = shift;
	$this->{'cnp_security'} = $value;
}


#################################################
#				Helper functions                #
#################################################

sub get_version {
	return "OpenECHO v1.0 06/26/2002";
}

sub getRandomCounter {
	srand( time() ^ ($$ + ($$ << 15)) );
	my $randnum = rand();
	$randnum = int($randnum * 1000000000);
	return $randnum;
}

sub get_EchoResponse {
	my $this = shift;
	return $this->{'EchoResponse'};
}

sub get_echotype1 {
	my $this = shift;
	return $this->{'echotype1'};
}

sub get_echotype2 {
	my $this = shift;
	return $this->{'echotype2'};
}

sub get_echotype3 {
	my $this = shift;
	return $this->{'echotype3'};
}

sub get_openecho {
	my $this = shift;
	return $this->{'openecho'};
}

sub set_EchoServer {
	my $this = shift;
	my $value = shift;
	$this->{'EchoServer'} = $value;
}

sub get_authorization {
	my $this = shift;
	return $this->{'authorization'};
}

sub get_avs_result {
	my $this = shift;
	return $this->{'avs_result'};
}

sub get_reference {
	my $this = shift;
	return $this->{'reference'};
}

sub get_order_number {
	my $this = shift;
	return $this->{'order_number'};
}

sub get_EchoSuccess {
	my $this = shift;
	return $this->{'EchoSuccess'};
}

sub get_status {
	my $this = shift;
	return $this->{'status'};
}

sub get_security_result {
	my $this = shift;
	return $this->GetEchoProp($this->{'echotype3'}, "security_result");
}

sub get_mac {
	my $this = shift;
	return $this->GetEchoProp($this->{'echotype3'}, "mac");
}

sub GetEchoProp {
	my $this = shift;
	local $_ = shift;
	lc;					# Make is lower case
	my $prop = shift;
	my $data;

	($data) = $_ =~ m"<$prop>(.*?)</$prop>"gsi;
	$data =~ s"<.*?>" "gs;
	chomp $data;
	return $data;
}

# Get's a given Echo return type and strips all HTML style tags from it.
# It also strips any new line characters from the returned string.
#
# This function based on Ben Reser's <breser@vecdev.com> Echo::Process
# module.
sub GetEchoReturn {
	my $this = shift;
	local $_ = shift;
	my $type = shift;
	my $data;

	if ($type eq 'OPEN') {
		($data) = $_ =~ m"<OPENECHO>(.*?)</OPENECHO>"gsi;
	}
	else {
		($data) = $_ =~ m"<ECHOTYPE$type>(.*?)</ECHOTYPE$type>"gsi;
	}
#	$data =~ s"<.*?>" "g;
	chomp $data;
	return $data;
}

sub _initialize {
	my $this = shift;

	$this->{'order_type'};
	$this->{'transaction_type'};
	$this->{'merchant_echo_id'};
	$this->{'merchant_pin'};
	$this->{'isp_echo_id'};
	$this->{'isp_pin'};
	$this->{'authorization'};
	$this->{'auth_code'};
	$this->{'billing_ip_address'};
	$this->{'billing_prefix'};
	# as of v1.5 billing_name is no longer being
	# used by ECHO use billing_first_name,
	# billing_last_name, and billing_company_name instead
	$this->{'billing_name'};
	$this->{'billing_first_name'};
	$this->{'billing_last_name'};
	$this->{'billing_company_name'};
	$this->{'billing_address1'};
	$this->{'billing_address2'};
	$this->{'billing_city'};
	$this->{'billing_state'};
	$this->{'billing_zip'};
	$this->{'billing_country'};
	$this->{'billing_phone'};
	$this->{'billing_fax'};
	$this->{'billing_email'};
	$this->{'cc_number'};
	$this->{'ccexp_month'};
	$this->{'ccexp_year'};
	$this->{'counter'};
	$this->{'debug'};
	$this->{'ec_account'};
	$this->{'ec_account_type'};
	$this->{'ec_payment_type'};
	$this->{'ec_address1'};
	$this->{'ec_address2'};
	$this->{'ec_bank_name'};
	$this->{'ec_business_acct'};
	$this->{'ec_city'};
	$this->{'ec_email'};
	$this->{'ec_first_name'};
	$this->{'ec_id_country'};
	$this->{'ec_id_exp_mm'};
	$this->{'ec_id_exp_dd'};
	$this->{'ec_id_exp_yy'};
	$this->{'ec_id_number'};
	$this->{'ec_id_state'};
	$this->{'ec_id_type'};
	$this->{'ec_last_name'};
	$this->{'ec_merchant_ref'};
	$this->{'ec_nbds_code'};
	$this->{'ec_other_name'};
	$this->{'ec_payee'};
	$this->{'ec_rt'};
	$this->{'ec_serial_number'};
	$this->{'ec_state'};
	$this->{'ec_transaction_dt'};
	$this->{'ec_zip'};
	$this->{'grand_total'};
	$this->{'merchant_email'};
	$this->{'merchant_trace_nbr'};
	$this->{'original_amount'};
	$this->{'original_trandate_mm'};
	$this->{'original_trandate_dd'};
	$this->{'original_trandate_yyyy'};
	$this->{'original_reference'};
	$this->{'order_number'};
	$this->{'shipping_flag'};
	$this->{'status'};
	$this->{'shipping_prefix'};
	$this->{'shipping_name'};
	$this->{'shipping_address1'};
	$this->{'shipping_address2'};
	$this->{'shipping_city'};
	$this->{'shipping_state'};
	$this->{'shipping_zip'};
	$this->{'shipping_comments'};
	$this->{'shipping_country'};
	$this->{'shipping_phone'};
	$this->{'shipping_fax'};
	$this->{'shipper'};
	$this->{'shipper_tracking_nbr'};
	$this->{'track1'};
	$this->{'track2'};
	$this->{'EchoResponse'};
	$this->{'EchoServer'};
	$this->{'echotype1'};
	$this->{'echotype2'};
	$this->{'echotype3'};
	$this->{'openecho'};
	$this->{'avs_result'};
	$this->{'athorization'};
	$this->{'reference'};
	$this->{'EchoSuccess'};
	$this->{'cnp_recurring'};
	$this->{'cnp_security'};
	$this->{'security_result'};
	$this->{'mac'};
	$this->{'cs_factors'};
	$this->{'cs_flag'};
	$this->{'cs_host_score'};
	$this->{'cs_reference_number'};
	$this->{'cs_response'};
	$this->{'cs_score'};
	$this->{'cs_status'};
}

1;

=head1 ECHO Transaction Module

OpenECHO

=head1 SYNOPSIS

use OpenECHO;
$openecho = new OpenECHO;

$openecho->set_EchoServer("https://wwws.echo-inc.com/scripts/INR200.EXE");
$openecho->set_transaction_type("CK");
$openecho->set_order_type("S");
$openecho->Submit();

print("<xmp>$openecho->echotype3<BR></xmp>\n");
print($openecho->{'EchoResponse'}."\n");

=head1 PREREQUISITES

  Net::SSLeay
 
    or
  
  LWP::UserAgent and Crypt::SSLeay

Only one of these need be present and working.  Net::SSLeay is preferred as some have reported problems using LWP::UserAgent and Crypt::SSLeay.

  URL::Escape

This module is used to write some of the URLs used by the OpenECHO module.

=head1 DESCRIPTION

The OpenECHO module.

=head2 Troubleshooting

=over 4

=item *

Make sure either Net::SSLeay or Crypt::SSLeay and LWP::UserAgent are installed
and working. You can test to see whether your Perl thinks they are:

    perl -MNet::SSLeay -e 'print "It works\n"'

or

    perl -MLWP::UserAgent -MCrypt::SSLeay -e 'print "It works\n"'

If either one prints "It works." and returns to the prompt you should be OK
(presuming they are in working order otherwise).

This module also uses URI::Escape;

=item *

Make sure you set your payment parameters properly.  

=back

=head1 AUTHOR

Michael Lehmkuhl <michael@electricpulp.com>

=head1 CREDITS

Jim Darden <support@openecho.com>

=cut
